function [IdentifiedPeaks]=idpeaktable(P,maxerror,Positions,Names,Intensities)
% Compares the measured peak positions in peak table "P" to a database of
% known peaks, in the form of an cell array of known peak maximum positions
% ("Positions") and matching cell array of names ("Names"). If the wavelength
% of a line in the database is closer to a measured peak position by less
% than the specified maximun error ("maxerror"), that peak is considered a
% match and its peak position, name, error, and amplitude are entered into
% the output cell array "IdentifiedPeaks" alopng with the wavleength and 
% relative intensity of each matching peak.
% For information on cell arrays, see
% http://www.mathworks.com/help/matlab/cell-arrays.html
%
% Use "cell2mat" to access numeric elements of IdentifiedPeaks, e.g.
% cell2mat(IdentifiedPeaks(2,1)) returns the position of the first 
% identified peak, cell2mat(IdentifiedPeaks(2,2)) returns its name, etc.
% Version 1.2, February 18, 2015, by Tom O'Haver (toh@umd.edu)
%
% Real data example: 
% >> load P
% >> load DataTableCu
% >> idpeaktable(P,.01,Positions,Names)
%
% This example loads the peak table "P", loads the database "DataTable"
% (containing an array of known peak maximum positions "Positions" and
% matching cell array of names "Names"), lists any peak in Cu that matches
% a known peaks in Positions within an error tolerance of 0.01 nm, and
% returns a table of identified peaks.
%
% Related function: idpeaks.m
%
%  Column labels for Named Peaks table 
IdentifiedPeaks(1,1)={'Position'};
IdentifiedPeaks(1,2)={'Name'};
IdentifiedPeaks(1,3)={'Wavelength'};
IdentifiedPeaks(1,4)={'Error'};
IdentifiedPeaks(1,5)={'Height'};
IdentifiedPeaks(1,6)={'Rel. Int.'};
p=2;
for n=1:length(Positions), % Look at each peak in database
   % m=index of the cloest match in Positions
   m=val2ind(P(:,2),Positions(n));
   % PositionError=difference between detected peak and nearest peak in table
   PositionError=abs(P(m,2)-Positions(n));
   if PositionError<maxerror, % Only identify the peaks if the error is less than MaxError
       % Assemble indentified peaks into cell array "IdentifiedPeaks"
       IdentifiedPeaks(p,:)=([P(m,2) Names(n) Positions(n) P(m,2)-Positions(n) P(m,3) Intensities(n)]);
       p=p+1;
   end % if PositionError
end  % for n
% ----------------------------------------------------------------------
function [index,closestval]=val2ind(x,val)
% Returns the index and the value of the element of vector x that is closest to val
% If more than one element is equally close, returns vectors of indicies and values
% Tom O'Haver (toh@umd.edu) October 2006
% Examples: If x=[1 2 4 3 5 9 6 4 5 3 1], then val2ind(x,6)=7 and val2ind(x,5.1)=[5 9]
% [indices values]=val2ind(x,3.3) returns indices = [4 10] and values = [3 3]
dif=abs(x-val);
index=find((dif-min(dif))==0);
closestval=x(index);


